import { useState, useMemo } from 'react';
import { useSearchParams } from 'react-router-dom';
import { SlidersHorizontal, Grid3X3, LayoutList, X } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Slider } from '@/components/ui/slider';
import { Checkbox } from '@/components/ui/checkbox';
import { Badge } from '@/components/ui/badge';
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from '@/components/ui/select';
import {
  Sheet,
  SheetContent,
  SheetHeader,
  SheetTitle,
  SheetTrigger,
} from '@/components/ui/sheet';
import { ProductCard } from '@/components/ProductCard';
import { useLanguage } from '@/contexts/LanguageContext';
import { useData } from '@/contexts/DataContext';
import type { Product } from '@/types';

type SortOption = 'newest' | 'price-asc' | 'price-desc' | 'rating' | 'discount';

export function Products() {
  const { language, t, dir } = useLanguage();
  const { products, categories } = useData();
  const [searchParams, setSearchParams] = useSearchParams();
  const [viewMode, setViewMode] = useState<'grid' | 'list'>('grid');
  const [isFilterOpen, setIsFilterOpen] = useState(false);

  // Filters
  const [selectedCategories, setSelectedCategories] = useState<string[]>([]);
  const [priceRange, setPriceRange] = useState<[number, number]>([0, 100000000]);
  const [minRating, setMinRating] = useState<number>(0);
  const [onlyDiscounted, setOnlyDiscounted] = useState(false);
  const [onlyInStock, setOnlyInStock] = useState(false);
  const [sortBy, setSortBy] = useState<SortOption>('newest');

  // Get search query from URL
  const searchQuery = searchParams.get('search') || '';
  const categoryFilter = searchParams.get('category') || '';
  const featuredFilter = searchParams.get('featured') === 'true';
  const newFilter = searchParams.get('new') === 'true';
  const discountFilter = searchParams.get('discount') === 'true';

  // Filter and sort products
  const filteredProducts = useMemo(() => {
    let result = [...products];

    // Search filter
    if (searchQuery) {
      const query = searchQuery.toLowerCase();
      result = result.filter(
        (p) =>
          p.name.toLowerCase().includes(query) ||
          p.nameEn.toLowerCase().includes(query) ||
          p.description.toLowerCase().includes(query)
      );
    }

    // Category filter
    if (categoryFilter) {
      result = result.filter((p) => p.category === categoryFilter || p.subcategory === categoryFilter);
    }

    // Selected categories filter
    if (selectedCategories.length > 0) {
      result = result.filter((p) => selectedCategories.includes(p.category));
    }

    // Featured filter
    if (featuredFilter) {
      result = result.filter((p) => p.isFeatured);
    }

    // New filter
    if (newFilter) {
      result = result.filter((p) => p.isNew);
    }

    // Discount filter
    if (discountFilter || onlyDiscounted) {
      result = result.filter((p) => p.discount && p.discount > 0);
    }

    // Price range filter
    result = result.filter((p) => p.price >= priceRange[0] && p.price <= priceRange[1]);

    // Rating filter
    if (minRating > 0) {
      result = result.filter((p) => p.rating >= minRating);
    }

    // Stock filter
    if (onlyInStock) {
      result = result.filter((p) => p.stock > 0);
    }

    // Sort
    switch (sortBy) {
      case 'price-asc':
        result.sort((a, b) => a.price - b.price);
        break;
      case 'price-desc':
        result.sort((a, b) => b.price - a.price);
        break;
      case 'rating':
        result.sort((a, b) => b.rating - a.rating);
        break;
      case 'discount':
        result.sort((a, b) => (b.discount || 0) - (a.discount || 0));
        break;
      case 'newest':
      default:
        result.sort((a, b) => new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime());
    }

    return result;
  }, [products, searchQuery, categoryFilter, selectedCategories, featuredFilter, newFilter, discountFilter, priceRange, minRating, onlyDiscounted, onlyInStock, sortBy]);

  const activeCategories = categories.filter((c) => c.isActive);

  const clearFilters = () => {
    setSelectedCategories([]);
    setPriceRange([0, 100000000]);
    setMinRating(0);
    setOnlyDiscounted(false);
    setOnlyInStock(false);
    setSearchParams({});
  };

  const hasActiveFilters =
    selectedCategories.length > 0 ||
    priceRange[0] > 0 ||
    priceRange[1] < 100000000 ||
    minRating > 0 ||
    onlyDiscounted ||
    onlyInStock ||
    searchQuery ||
    categoryFilter;

  const FilterContent = () => (
    <div className="space-y-6">
      {/* Categories */}
      <div>
        <h4 className="font-semibold mb-3">{t('دسته‌بندی‌ها', 'Categories')}</h4>
        <div className="space-y-2">
          {activeCategories.map((category) => (
            <div key={category.id} className="flex items-center gap-2">
              <Checkbox
                id={`cat-${category.id}`}
                checked={selectedCategories.includes(category.id)}
                onCheckedChange={(checked) => {
                  if (checked) {
                    setSelectedCategories([...selectedCategories, category.id]);
                  } else {
                    setSelectedCategories(selectedCategories.filter((id) => id !== category.id));
                  }
                }}
              />
              <label htmlFor={`cat-${category.id}`} className="text-sm cursor-pointer">
                {language === 'fa' ? category.name : category.nameEn}
              </label>
            </div>
          ))}
        </div>
      </div>

      {/* Price Range */}
      <div>
        <h4 className="font-semibold mb-3">{t('محدوده قیمت', 'Price Range')}</h4>
        <Slider
          value={priceRange}
          onValueChange={(value) => setPriceRange(value as [number, number])}
          max={100000000}
          step={1000000}
          className="mb-4"
        />
        <div className="flex justify-between text-sm">
          <span>{priceRange[0].toLocaleString()}</span>
          <span>{priceRange[1].toLocaleString()}</span>
        </div>
      </div>

      {/* Rating */}
      <div>
        <h4 className="font-semibold mb-3">{t('حداقل امتیاز', 'Minimum Rating')}</h4>
        <div className="space-y-2">
          {[4, 3, 2, 1].map((rating) => (
            <div key={rating} className="flex items-center gap-2">
              <input
                type="radio"
                name="rating"
                id={`rating-${rating}`}
                checked={minRating === rating}
                onChange={() => setMinRating(rating)}
                className="cursor-pointer"
              />
              <label htmlFor={`rating-${rating}`} className="text-sm cursor-pointer flex items-center gap-1">
                {rating}+ {t('ستاره', 'stars')}
              </label>
            </div>
          ))}
          <div className="flex items-center gap-2">
            <input
              type="radio"
              name="rating"
              id="rating-0"
              checked={minRating === 0}
              onChange={() => setMinRating(0)}
              className="cursor-pointer"
            />
            <label htmlFor="rating-0" className="text-sm cursor-pointer">
              {t('همه', 'All')}
            </label>
          </div>
        </div>
      </div>

      {/* Other Filters */}
      <div className="space-y-2">
        <div className="flex items-center gap-2">
          <Checkbox
            id="discount"
            checked={onlyDiscounted}
            onCheckedChange={(checked) => setOnlyDiscounted(checked as boolean)}
          />
          <label htmlFor="discount" className="text-sm cursor-pointer">
            {t('فقط تخفیف‌دارها', 'Only discounted')}
          </label>
        </div>
        <div className="flex items-center gap-2">
          <Checkbox
            id="instock"
            checked={onlyInStock}
            onCheckedChange={(checked) => setOnlyInStock(checked as boolean)}
          />
          <label htmlFor="instock" className="text-sm cursor-pointer">
            {t('فقط موجودها', 'Only in stock')}
          </label>
        </div>
      </div>

      <Button variant="outline" className="w-full" onClick={clearFilters}>
        {t('پاک کردن فیلترها', 'Clear Filters')}
      </Button>
    </div>
  );

  return (
    <div className="min-h-screen py-8">
      <div className="container mx-auto px-4">
        {/* Header */}
        <div className="mb-8">
          <h1 className="text-2xl md:text-3xl font-bold mb-2">
            {searchQuery
              ? `${t('نتایج جستجو برای', 'Search results for')}: "${searchQuery}"`
              : categoryFilter
              ? language === 'fa'
                ? categories.find((c) => c.id === categoryFilter)?.name
                : categories.find((c) => c.id === categoryFilter)?.nameEn
              : t('همه محصولات', 'All Products')}
          </h1>
          <p className="text-muted-foreground">
            {filteredProducts.length} {t('محصول', 'product')}
          </p>
        </div>

        {/* Toolbar */}
        <div className="flex flex-wrap items-center justify-between gap-4 mb-6">
          <div className="flex items-center gap-2">
            {/* Mobile Filter */}
            <Sheet open={isFilterOpen} onOpenChange={setIsFilterOpen}>
              <SheetTrigger asChild>
                <Button variant="outline" className="lg:hidden">
                  <SlidersHorizontal className="h-4 w-4 ml-2" />
                  {t('فیلترها', 'Filters')}
                </Button>
              </SheetTrigger>
              <SheetContent side={dir === 'rtl' ? 'right' : 'left'} className="w-80">
                <SheetHeader>
                  <SheetTitle>{t('فیلترها', 'Filters')}</SheetTitle>
                </SheetHeader>
                <div className="mt-6">
                  <FilterContent />
                </div>
              </SheetContent>
            </Sheet>

            {hasActiveFilters && (
              <Button variant="ghost" size="sm" onClick={clearFilters}>
                <X className="h-4 w-4 ml-2" />
                {t('پاک کردن', 'Clear')}
              </Button>
            )}
          </div>

          <div className="flex items-center gap-4">
            {/* Sort */}
            <Select value={sortBy} onValueChange={(value) => setSortBy(value as SortOption)}>
              <SelectTrigger className="w-40">
                <SelectValue placeholder={t('مرتب‌سازی', 'Sort')} />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="newest">{t('جدیدترین', 'Newest')}</SelectItem>
                <SelectItem value="price-asc">{t('قیمت: کم به زیاد', 'Price: Low to High')}</SelectItem>
                <SelectItem value="price-desc">{t('قیمت: زیاد به کم', 'Price: High to Low')}</SelectItem>
                <SelectItem value="rating">{t('بیشترین امتیاز', 'Highest Rated')}</SelectItem>
                <SelectItem value="discount">{t('بیشترین تخفیف', 'Biggest Discount')}</SelectItem>
              </SelectContent>
            </Select>

            {/* View Mode */}
            <div className="hidden sm:flex items-center border rounded-md">
              <Button
                variant={viewMode === 'grid' ? 'default' : 'ghost'}
                size="icon"
                className="rounded-none rounded-l-md"
                onClick={() => setViewMode('grid')}
              >
                <Grid3X3 className="h-4 w-4" />
              </Button>
              <Button
                variant={viewMode === 'list' ? 'default' : 'ghost'}
                size="icon"
                className="rounded-none rounded-r-md"
                onClick={() => setViewMode('list')}
              >
                <LayoutList className="h-4 w-4" />
              </Button>
            </div>
          </div>
        </div>

        {/* Active Filters */}
        {hasActiveFilters && (
          <div className="flex flex-wrap gap-2 mb-6">
            {searchQuery && (
              <Badge variant="secondary" className="gap-1">
                {t('جستجو', 'Search')}: {searchQuery}
                <X className="h-3 w-3 cursor-pointer" onClick={() => setSearchParams({})} />
              </Badge>
            )}
            {selectedCategories.map((catId) => {
              const cat = categories.find((c) => c.id === catId);
              return (
                <Badge key={catId} variant="secondary" className="gap-1">
                  {language === 'fa' ? cat?.name : cat?.nameEn}
                  <X
                    className="h-3 w-3 cursor-pointer"
                    onClick={() => setSelectedCategories(selectedCategories.filter((id) => id !== catId))}
                  />
                </Badge>
              );
            })}
            {(priceRange[0] > 0 || priceRange[1] < 100000000) && (
              <Badge variant="secondary" className="gap-1">
                {t('قیمت', 'Price')}: {priceRange[0].toLocaleString()} - {priceRange[1].toLocaleString()}
                <X className="h-3 w-3 cursor-pointer" onClick={() => setPriceRange([0, 100000000])} />
              </Badge>
            )}
            {minRating > 0 && (
              <Badge variant="secondary" className="gap-1">
                {t('امتیاز', 'Rating')}: {minRating}+
                <X className="h-3 w-3 cursor-pointer" onClick={() => setMinRating(0)} />
              </Badge>
            )}
          </div>
        )}

        {/* Content */}
        <div className="flex gap-8">
          {/* Sidebar - Desktop */}
          <aside className="hidden lg:block w-64 flex-shrink-0">
            <div className="sticky top-24">
              <h3 className="font-bold text-lg mb-4">{t('فیلترها', 'Filters')}</h3>
              <FilterContent />
            </div>
          </aside>

          {/* Products Grid */}
          <div className="flex-1">
            {filteredProducts.length === 0 ? (
              <div className="text-center py-16">
                <p className="text-muted-foreground text-lg">
                  {t('محصولی یافت نشد', 'No products found')}
                </p>
                <Button variant="outline" className="mt-4" onClick={clearFilters}>
                  {t('پاک کردن فیلترها', 'Clear Filters')}
                </Button>
              </div>
            ) : (
              <div
                className={`grid gap-6 ${
                  viewMode === 'grid'
                    ? 'grid-cols-1 sm:grid-cols-2 xl:grid-cols-3'
                    : 'grid-cols-1'
                }`}
              >
                {filteredProducts.map((product) => (
                  <ProductCard key={product.id} product={product} />
                ))}
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  );
}
