import React, { createContext, useContext, useState, useEffect } from 'react';
import type { User } from '@/types';
import { toast } from 'sonner';
import { useLanguage } from './LanguageContext';

interface AuthContextType {
  user: User | null;
  isAuthenticated: boolean;
  isAdmin: boolean;
  login: (email: string, password: string) => Promise<boolean>;
  register: (name: string, email: string, password: string) => Promise<boolean>;
  logout: () => void;
  updateProfile: (data: Partial<User>) => void;
}

const AuthContext = createContext<AuthContextType | undefined>(undefined);

// Default admin account
const DEFAULT_ADMIN = {
  id: 'admin-1',
  name: 'مدیر سیستم',
  email: 'admin@digitalshop.ir',
  password: 'admin123',
  isAdmin: true,
  createdAt: new Date().toISOString(),
};

export function AuthProvider({ children }: { children: React.ReactNode }) {
  const { t } = useLanguage();
  const [user, setUser] = useState<User | null>(() => {
    if (typeof window !== 'undefined') {
      const saved = localStorage.getItem('user');
      return saved ? JSON.parse(saved) : null;
    }
    return null;
  });

  // Initialize default admin in localStorage
  useEffect(() => {
    const users = JSON.parse(localStorage.getItem('users') || '[]');
    if (!users.find((u: any) => u.email === DEFAULT_ADMIN.email)) {
      users.push(DEFAULT_ADMIN);
      localStorage.setItem('users', JSON.stringify(users));
    }
  }, []);

  useEffect(() => {
    if (user) {
      localStorage.setItem('user', JSON.stringify(user));
    } else {
      localStorage.removeItem('user');
    }
  }, [user]);

  const login = async (email: string, password: string): Promise<boolean> => {
    const users = JSON.parse(localStorage.getItem('users') || '[]');
    const found = users.find((u: any) => u.email === email && u.password === password);
    
    if (found) {
      const { password, ...userWithoutPassword } = found;
      setUser(userWithoutPassword);
      toast.success(t('ورود موفقیت‌آمیز', 'Login successful'));
      return true;
    }
    
    toast.error(t('ایمیل یا رمز عبور اشتباه است', 'Invalid email or password'));
    return false;
  };

  const register = async (name: string, email: string, password: string): Promise<boolean> => {
    const users = JSON.parse(localStorage.getItem('users') || '[]');
    
    if (users.find((u: any) => u.email === email)) {
      toast.error(t('این ایمیل قبلاً ثبت شده است', 'This email is already registered'));
      return false;
    }

    const newUser = {
      id: `user-${Date.now()}`,
      name,
      email,
      password,
      isAdmin: false,
      createdAt: new Date().toISOString(),
    };

    users.push(newUser);
    localStorage.setItem('users', JSON.stringify(users));

    const { password: _, ...userWithoutPassword } = newUser;
    setUser(userWithoutPassword);
    toast.success(t('ثبت‌نام موفقیت‌آمیز', 'Registration successful'));
    return true;
  };

  const logout = () => {
    setUser(null);
    toast.success(t('خروج موفقیت‌آمیز', 'Logout successful'));
  };

  const updateProfile = (data: Partial<User>) => {
    if (user) {
      const updated = { ...user, ...data };
      setUser(updated);
      
      // Update in users list
      const users = JSON.parse(localStorage.getItem('users') || '[]');
      const index = users.findIndex((u: any) => u.id === user.id);
      if (index !== -1) {
        users[index] = { ...users[index], ...data };
        localStorage.setItem('users', JSON.stringify(users));
      }
      toast.success(t('پروفایل به‌روزرسانی شد', 'Profile updated'));
    }
  };

  return (
    <AuthContext.Provider
      value={{
        user,
        isAuthenticated: !!user,
        isAdmin: user?.isAdmin || false,
        login,
        register,
        logout,
        updateProfile,
      }}
    >
      {children}
    </AuthContext.Provider>
  );
}

export function useAuth() {
  const context = useContext(AuthContext);
  if (context === undefined) {
    throw new Error('useAuth must be used within an AuthProvider');
  }
  return context;
}
