import { Link } from 'react-router-dom';
import {
  Instagram,
  Send,
  Phone,
  Mail,
  MapPin,
  CreditCard,
  Truck,
  Shield,
  Headphones,
} from 'lucide-react';
import { useLanguage } from '@/contexts/LanguageContext';
import { useData } from '@/contexts/DataContext';

export function Footer() {
  const { t, language } = useLanguage();
  const { siteSettings, categories } = useData();

  const activeCategories = categories.filter((c) => c.isActive).slice(0, 6);

  return (
    <footer className="bg-muted mt-auto">
      {/* Features Bar */}
      <div className="border-b">
        <div className="container mx-auto px-4 py-8">
          <div className="grid grid-cols-2 md:grid-cols-4 gap-6">
            <div className="flex items-center gap-3">
              <div className="w-12 h-12 bg-primary/10 rounded-full flex items-center justify-center flex-shrink-0">
                <Truck className="h-6 w-6 text-primary" />
              </div>
              <div>
                <h4 className="font-semibold text-sm">{t('ارسال رایگان', 'Free Shipping')}</h4>
                <p className="text-xs text-muted-foreground">{t('برای خرید بالای 5 میلیون', 'Over 50M')}</p>
              </div>
            </div>
            <div className="flex items-center gap-3">
              <div className="w-12 h-12 bg-primary/10 rounded-full flex items-center justify-center flex-shrink-0">
                <Shield className="h-6 w-6 text-primary" />
              </div>
              <div>
                <h4 className="font-semibold text-sm">{t('ضمانت اصالت', 'Authenticity')}</h4>
                <p className="text-xs text-muted-foreground">{t('100% اورجینال', '100% Original')}</p>
              </div>
            </div>
            <div className="flex items-center gap-3">
              <div className="w-12 h-12 bg-primary/10 rounded-full flex items-center justify-center flex-shrink-0">
                <CreditCard className="h-6 w-6 text-primary" />
              </div>
              <div>
                <h4 className="font-semibold text-sm">{t('پرداخت امن', 'Secure Payment')}</h4>
                <p className="text-xs text-muted-foreground">{t('درگاه معتبر', 'Trusted Gateway')}</p>
              </div>
            </div>
            <div className="flex items-center gap-3">
              <div className="w-12 h-12 bg-primary/10 rounded-full flex items-center justify-center flex-shrink-0">
                <Headphones className="h-6 w-6 text-primary" />
              </div>
              <div>
                <h4 className="font-semibold text-sm">{t('پشتیبانی 24/7', '24/7 Support')}</h4>
                <p className="text-xs text-muted-foreground">{t('همیشه در دسترس', 'Always Available')}</p>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Main Footer */}
      <div className="container mx-auto px-4 py-12">
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
          {/* About */}
          <div>
            <div className="flex items-center gap-2 mb-4">
              <div className="w-10 h-10 bg-primary rounded-lg flex items-center justify-center">
                <span className="text-primary-foreground font-bold text-xl">D</span>
              </div>
              <span className="font-bold text-xl">
                {language === 'fa' ? siteSettings.siteName : siteSettings.siteNameEn}
              </span>
            </div>
            <p className="text-sm text-muted-foreground mb-4">
              {language === 'fa' ? siteSettings.description : siteSettings.descriptionEn}
            </p>
            <div className="space-y-2">
              <div className="flex items-center gap-2 text-sm">
                <Phone className="h-4 w-4 text-primary" />
                <span>{siteSettings.phone}</span>
              </div>
              <div className="flex items-center gap-2 text-sm">
                <Mail className="h-4 w-4 text-primary" />
                <span>{siteSettings.email}</span>
              </div>
              <div className="flex items-center gap-2 text-sm">
                <MapPin className="h-4 w-4 text-primary" />
                <span>{siteSettings.address}</span>
              </div>
            </div>
          </div>

          {/* Categories */}
          <div>
            <h3 className="font-bold text-lg mb-4">{t('دسته‌بندی‌ها', 'Categories')}</h3>
            <ul className="space-y-2">
              {activeCategories.map((category) => (
                <li key={category.id}>
                  <Link
                    to={`/category/${category.id}`}
                    className="text-sm text-muted-foreground hover:text-primary transition-colors"
                  >
                    {language === 'fa' ? category.name : category.nameEn}
                  </Link>
                </li>
              ))}
            </ul>
          </div>

          {/* Quick Links */}
          <div>
            <h3 className="font-bold text-lg mb-4">{t('دسترسی سریع', 'Quick Links')}</h3>
            <ul className="space-y-2">
              <li>
                <Link to="/products" className="text-sm text-muted-foreground hover:text-primary transition-colors">
                  {t('همه محصولات', 'All Products')}
                </Link>
              </li>
              <li>
                <Link to="/blog" className="text-sm text-muted-foreground hover:text-primary transition-colors">
                  {t('وبلاگ', 'Blog')}
                </Link>
              </li>
              <li>
                <Link to="/about" className="text-sm text-muted-foreground hover:text-primary transition-colors">
                  {t('درباره ما', 'About Us')}
                </Link>
              </li>
              <li>
                <Link to="/contact" className="text-sm text-muted-foreground hover:text-primary transition-colors">
                  {t('تماس با ما', 'Contact')}
                </Link>
              </li>
              <li>
                <Link to="/faq" className="text-sm text-muted-foreground hover:text-primary transition-colors">
                  {t('سوالات متداول', 'FAQ')}
                </Link>
              </li>
              <li>
                <Link to="/terms" className="text-sm text-muted-foreground hover:text-primary transition-colors">
                  {t('قوانین و مقررات', 'Terms & Conditions')}
                </Link>
              </li>
            </ul>
          </div>

          {/* Social & Newsletter */}
          <div>
            <h3 className="font-bold text-lg mb-4">{t('ما را دنبال کنید', 'Follow Us')}</h3>
            <div className="flex gap-3 mb-6">
              {siteSettings.socialLinks.instagram && (
                <a
                  href={siteSettings.socialLinks.instagram}
                  target="_blank"
                  rel="noopener noreferrer"
                  className="w-10 h-10 bg-primary/10 rounded-full flex items-center justify-center hover:bg-primary hover:text-primary-foreground transition-colors"
                >
                  <Instagram className="h-5 w-5" />
                </a>
              )}
              {siteSettings.socialLinks.telegram && (
                <a
                  href={siteSettings.socialLinks.telegram}
                  target="_blank"
                  rel="noopener noreferrer"
                  className="w-10 h-10 bg-primary/10 rounded-full flex items-center justify-center hover:bg-primary hover:text-primary-foreground transition-colors"
                >
                  <Send className="h-5 w-5" />
                </a>
              )}
              {siteSettings.socialLinks.whatsapp && (
                <a
                  href={siteSettings.socialLinks.whatsapp}
                  target="_blank"
                  rel="noopener noreferrer"
                  className="w-10 h-10 bg-primary/10 rounded-full flex items-center justify-center hover:bg-primary hover:text-primary-foreground transition-colors"
                >
                  <Phone className="h-5 w-5" />
                </a>
              )}
            </div>

            <h4 className="font-semibold mb-2">{t('عضویت در خبرنامه', 'Newsletter')}</h4>
            <p className="text-xs text-muted-foreground mb-3">
              {t('از جدیدترین تخفیف‌ها مطلع شوید', 'Get notified about latest discounts')}
            </p>
            <div className="flex gap-2">
              <input
                type="email"
                placeholder={t('ایمیل خود را وارد کنید', 'Enter your email')}
                className="flex-1 px-3 py-2 text-sm border rounded-md bg-background"
                dir={language === 'fa' ? 'rtl' : 'ltr'}
              />
              <button className="px-4 py-2 bg-primary text-primary-foreground rounded-md text-sm hover:bg-primary/90 transition-colors">
                {t('عضویت', 'Subscribe')}
              </button>
            </div>
          </div>
        </div>
      </div>

      {/* Copyright */}
      <div className="border-t">
        <div className="container mx-auto px-4 py-4">
          <div className="flex flex-col md:flex-row justify-between items-center gap-4">
            <p className="text-sm text-muted-foreground text-center md:text-left">
              {language === 'fa' ? siteSettings.footerText : siteSettings.footerTextEn} © {new Date().getFullYear()}
            </p>
            <div className="flex items-center gap-4">
              <img
                src="https://upload.wikimedia.org/wikipedia/commons/thumb/5/5e/Visa_Inc._logo.svg/100px-Visa_Inc._logo.svg.png"
                alt="Visa"
                className="h-6 opacity-50"
              />
              <img
                src="https://upload.wikimedia.org/wikipedia/commons/thumb/2/2a/Mastercard-logo.svg/100px-Mastercard-logo.svg.png"
                alt="Mastercard"
                className="h-6 opacity-50"
              />
            </div>
          </div>
        </div>
      </div>
    </footer>
  );
}
