import { useState } from 'react';
import { Plus, Edit, Trash2, GripVertical } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Checkbox } from '@/components/ui/checkbox';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from '@/components/ui/dialog';
import { useLanguage } from '@/contexts/LanguageContext';
import { useData } from '@/contexts/DataContext';
import type { Category } from '@/types';
import { toast } from 'sonner';

export function AdminCategories() {
  const { language, t, dir } = useLanguage();
  const { categories, addCategory, updateCategory, deleteCategory } = useData();
  const [editingCategory, setEditingCategory] = useState<Category | null>(null);
  const [isFormOpen, setIsFormOpen] = useState(false);

  const [formData, setFormData] = useState<Partial<Category>>({
    name: '',
    nameEn: '',
    description: '',
    descriptionEn: '',
    image: '',
    icon: 'Package',
    sortOrder: 0,
    isActive: true,
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();

    if (!formData.name || !formData.nameEn) {
      toast.error(t('لطفاً نام دسته‌بندی را وارد کنید', 'Please enter category name'));
      return;
    }

    if (editingCategory) {
      updateCategory(editingCategory.id, formData);
      toast.success(t('دسته‌بندی به‌روزرسانی شد', 'Category updated'));
    } else {
      addCategory(formData as Omit<Category, 'id'>);
      toast.success(t('دسته‌بندی جدید اضافه شد', 'Category added'));
    }

    setIsFormOpen(false);
    setEditingCategory(null);
    resetForm();
  };

  const resetForm = () => {
    setFormData({
      name: '',
      nameEn: '',
      description: '',
      descriptionEn: '',
      image: '',
      icon: 'Package',
      sortOrder: 0,
      isActive: true,
    });
  };

  const handleEdit = (category: Category) => {
    setEditingCategory(category);
    setFormData(category);
    setIsFormOpen(true);
  };

  const handleDelete = (id: string) => {
    if (confirm(t('آیا از حذف این دسته‌بندی مطمئن هستید؟', 'Are you sure you want to delete this category?'))) {
      deleteCategory(id);
      toast.success(t('دسته‌بندی حذف شد', 'Category deleted'));
    }
  };

  const sortedCategories = [...categories].sort((a, b) => a.sortOrder - b.sortOrder);

  return (
    <div className="space-y-6">
      <div className="flex flex-col sm:flex-row sm:items-center justify-between gap-4">
        <h1 className="text-2xl md:text-3xl font-bold">{t('مدیریت دسته‌بندی‌ها', 'Manage Categories')}</h1>
        <Dialog open={isFormOpen} onOpenChange={setIsFormOpen}>
          <DialogTrigger asChild>
            <Button
              onClick={() => {
                setEditingCategory(null);
                resetForm();
              }}
            >
              <Plus className="h-4 w-4 ml-2" />
              {t('افزودن دسته‌بندی', 'Add Category')}
            </Button>
          </DialogTrigger>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>
                {editingCategory ? t('ویرایش دسته‌بندی', 'Edit Category') : t('افزودن دسته‌بندی جدید', 'Add New Category')}
              </DialogTitle>
              <DialogDescription>
                {t('اطلاعات دسته‌بندی را وارد کنید', 'Enter category information')}
              </DialogDescription>
            </DialogHeader>
            <form onSubmit={handleSubmit} className="space-y-4" dir={dir}>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="name">{t('نام (فارسی)', 'Name (FA)')} *</Label>
                  <Input
                    id="name"
                    value={formData.name}
                    onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                    dir="rtl"
                    required
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="nameEn">{t('نام (انگلیسی)', 'Name (EN)')} *</Label>
                  <Input
                    id="nameEn"
                    value={formData.nameEn}
                    onChange={(e) => setFormData({ ...formData, nameEn: e.target.value })}
                    dir="ltr"
                    required
                  />
                </div>
              </div>

              <div className="space-y-2">
                <Label htmlFor="description">{t('توضیحات (فارسی)', 'Description (FA)')}</Label>
                <Input
                  id="description"
                  value={formData.description}
                  onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                  dir="rtl"
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="descriptionEn">{t('توضیحات (انگلیسی)', 'Description (EN)')}</Label>
                <Input
                  id="descriptionEn"
                  value={formData.descriptionEn}
                  onChange={(e) => setFormData({ ...formData, descriptionEn: e.target.value })}
                  dir="ltr"
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="image">{t('URL تصویر', 'Image URL')}</Label>
                <Input
                  id="image"
                  value={formData.image}
                  onChange={(e) => setFormData({ ...formData, image: e.target.value })}
                  dir="ltr"
                  placeholder="https://..."
                />
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="icon">{t('آیکون', 'Icon')}</Label>
                  <Input
                    id="icon"
                    value={formData.icon}
                    onChange={(e) => setFormData({ ...formData, icon: e.target.value })}
                    dir="ltr"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="sortOrder">{t('ترتیب', 'Sort Order')}</Label>
                  <Input
                    id="sortOrder"
                    type="number"
                    value={formData.sortOrder}
                    onChange={(e) => setFormData({ ...formData, sortOrder: Number(e.target.value) })}
                    dir="ltr"
                  />
                </div>
              </div>

              <div className="flex items-center gap-2">
                <Checkbox
                  id="isActive"
                  checked={formData.isActive}
                  onCheckedChange={(checked) => setFormData({ ...formData, isActive: checked as boolean })}
                />
                <Label htmlFor="isActive">{t('فعال', 'Active')}</Label>
              </div>

              <div className="flex justify-end gap-2">
                <Button type="button" variant="outline" onClick={() => setIsFormOpen(false)}>
                  {t('انصراف', 'Cancel')}
                </Button>
                <Button type="submit">
                  {editingCategory ? t('به‌روزرسانی', 'Update') : t('ذخیره', 'Save')}
                </Button>
              </div>
            </form>
          </DialogContent>
        </Dialog>
      </div>

      {/* Categories Table */}
      <div className="bg-card rounded-lg border overflow-hidden">
        <div className="overflow-x-auto">
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead className="w-10"></TableHead>
                <TableHead>{t('دسته‌بندی', 'Category')}</TableHead>
                <TableHead>{t('ترتیب', 'Order')}</TableHead>
                <TableHead>{t('وضعیت', 'Status')}</TableHead>
                <TableHead className="text-left">{t('عملیات', 'Actions')}</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {sortedCategories.length === 0 ? (
                <TableRow>
                  <TableCell colSpan={5} className="text-center py-8 text-muted-foreground">
                    {t('دسته‌بندی یافت نشد', 'No categories found')}
                  </TableCell>
                </TableRow>
              ) : (
                sortedCategories.map((category) => (
                  <TableRow key={category.id}>
                    <TableCell>
                      <GripVertical className="h-4 w-4 text-muted-foreground" />
                    </TableCell>
                    <TableCell>
                      <div className="flex items-center gap-3">
                        <div className="w-12 h-12 bg-muted rounded overflow-hidden">
                          <img
                            src={category.image}
                            alt={language === 'fa' ? category.name : category.nameEn}
                            className="w-full h-full object-cover"
                          />
                        </div>
                        <div>
                          <p className="font-medium">
                            {language === 'fa' ? category.name : category.nameEn}
                          </p>
                          <p className="text-xs text-muted-foreground">{category.icon}</p>
                        </div>
                      </div>
                    </TableCell>
                    <TableCell>{category.sortOrder}</TableCell>
                    <TableCell>
                      <span
                        className={`px-2 py-1 rounded text-xs ${
                          category.isActive
                            ? 'bg-green-100 text-green-800'
                            : 'bg-gray-100 text-gray-800'
                        }`}
                      >
                        {category.isActive ? t('فعال', 'Active') : t('غیرفعال', 'Inactive')}
                      </span>
                    </TableCell>
                    <TableCell>
                      <div className="flex items-center gap-1">
                        <Button variant="ghost" size="icon" onClick={() => handleEdit(category)}>
                          <Edit className="h-4 w-4" />
                        </Button>
                        <Button
                          variant="ghost"
                          size="icon"
                          className="text-destructive"
                          onClick={() => handleDelete(category.id)}
                        >
                          <Trash2 className="h-4 w-4" />
                        </Button>
                      </div>
                    </TableCell>
                  </TableRow>
                ))
              )}
            </TableBody>
          </Table>
        </div>
      </div>
    </div>
  );
}
