import { useState } from 'react';
import { Plus, Edit, Trash2, Eye, EyeOff } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Checkbox } from '@/components/ui/checkbox';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from '@/components/ui/dialog';
import { useLanguage } from '@/contexts/LanguageContext';
import { useData } from '@/contexts/DataContext';
import type { Banner } from '@/types';
import { toast } from 'sonner';

export function AdminBanners() {
  const { language, t, dir } = useLanguage();
  const { banners, addBanner, updateBanner, deleteBanner } = useData();
  const [editingBanner, setEditingBanner] = useState<Banner | null>(null);
  const [isFormOpen, setIsFormOpen] = useState(false);

  const [formData, setFormData] = useState({
    title: '',
    titleEn: '',
    subtitle: '',
    subtitleEn: '',
    image: '',
    link: '',
    buttonText: '',
    buttonTextEn: '',
    isActive: true,
    sortOrder: 0,
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();

    if (!formData.title || !formData.image) {
      toast.error(t('لطفاً عنوان و تصویر را وارد کنید', 'Please enter title and image'));
      return;
    }

    if (editingBanner) {
      updateBanner(editingBanner.id, formData);
      toast.success(t('بنر به‌روزرسانی شد', 'Banner updated'));
    } else {
      addBanner(formData);
      toast.success(t('بنر جدید اضافه شد', 'Banner added'));
    }

    setIsFormOpen(false);
    setEditingBanner(null);
    resetForm();
  };

  const resetForm = () => {
    setFormData({
      title: '',
      titleEn: '',
      subtitle: '',
      subtitleEn: '',
      image: '',
      link: '',
      buttonText: '',
      buttonTextEn: '',
      isActive: true,
      sortOrder: 0,
    });
  };

  const handleEdit = (banner: Banner) => {
    setEditingBanner(banner);
    setFormData(banner);
    setIsFormOpen(true);
  };

  const handleDelete = (id: string) => {
    if (confirm(t('آیا از حذف این بنر مطمئن هستید؟', 'Are you sure you want to delete this banner?'))) {
      deleteBanner(id);
      toast.success(t('بنر حذف شد', 'Banner deleted'));
    }
  };

  const sortedBanners = [...banners].sort((a, b) => a.sortOrder - b.sortOrder);

  return (
    <div className="space-y-6">
      <div className="flex flex-col sm:flex-row sm:items-center justify-between gap-4">
        <h1 className="text-2xl md:text-3xl font-bold">{t('مدیریت بنرها', 'Manage Banners')}</h1>
        <Dialog open={isFormOpen} onOpenChange={setIsFormOpen}>
          <DialogTrigger asChild>
            <Button
              onClick={() => {
                setEditingBanner(null);
                resetForm();
              }}
            >
              <Plus className="h-4 w-4 ml-2" />
              {t('افزودن بنر', 'Add Banner')}
            </Button>
          </DialogTrigger>
          <DialogContent className="max-w-2xl">
            <DialogHeader>
              <DialogTitle>
                {editingBanner ? t('ویرایش بنر', 'Edit Banner') : t('افزودن بنر جدید', 'Add New Banner')}
              </DialogTitle>
              <DialogDescription>
                {t('اطلاعات بنر را وارد کنید', 'Enter banner information')}
              </DialogDescription>
            </DialogHeader>
            <form onSubmit={handleSubmit} className="space-y-4" dir={dir}>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="title">{t('عنوان (فارسی)', 'Title (FA)')} *</Label>
                  <Input
                    id="title"
                    value={formData.title}
                    onChange={(e) => setFormData({ ...formData, title: e.target.value })}
                    dir="rtl"
                    required
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="titleEn">{t('عنوان (انگلیسی)', 'Title (EN)')}</Label>
                  <Input
                    id="titleEn"
                    value={formData.titleEn}
                    onChange={(e) => setFormData({ ...formData, titleEn: e.target.value })}
                    dir="ltr"
                  />
                </div>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="subtitle">{t('زیرعنوان (فارسی)', 'Subtitle (FA)')}</Label>
                  <Input
                    id="subtitle"
                    value={formData.subtitle}
                    onChange={(e) => setFormData({ ...formData, subtitle: e.target.value })}
                    dir="rtl"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="subtitleEn">{t('زیرعنوان (انگلیسی)', 'Subtitle (EN)')}</Label>
                  <Input
                    id="subtitleEn"
                    value={formData.subtitleEn}
                    onChange={(e) => setFormData({ ...formData, subtitleEn: e.target.value })}
                    dir="ltr"
                  />
                </div>
              </div>

              <div className="space-y-2">
                <Label htmlFor="image">{t('URL تصویر', 'Image URL')} *</Label>
                <Input
                  id="image"
                  value={formData.image}
                  onChange={(e) => setFormData({ ...formData, image: e.target.value })}
                  dir="ltr"
                  placeholder="https://..."
                  required
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="link">{t('لینک', 'Link')}</Label>
                <Input
                  id="link"
                  value={formData.link}
                  onChange={(e) => setFormData({ ...formData, link: e.target.value })}
                  dir="ltr"
                  placeholder="/products"
                />
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="buttonText">{t('متن دکمه (فارسی)', 'Button Text (FA)')}</Label>
                  <Input
                    id="buttonText"
                    value={formData.buttonText}
                    onChange={(e) => setFormData({ ...formData, buttonText: e.target.value })}
                    dir="rtl"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="buttonTextEn">{t('متن دکمه (انگلیسی)', 'Button Text (EN)')}</Label>
                  <Input
                    id="buttonTextEn"
                    value={formData.buttonTextEn}
                    onChange={(e) => setFormData({ ...formData, buttonTextEn: e.target.value })}
                    dir="ltr"
                  />
                </div>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="sortOrder">{t('ترتیب', 'Sort Order')}</Label>
                  <Input
                    id="sortOrder"
                    type="number"
                    value={formData.sortOrder}
                    onChange={(e) => setFormData({ ...formData, sortOrder: Number(e.target.value) })}
                    dir="ltr"
                  />
                </div>
                <div className="flex items-center gap-2 pt-8">
                  <Checkbox
                    id="isActive"
                    checked={formData.isActive}
                    onCheckedChange={(checked) =>
                      setFormData({ ...formData, isActive: checked as boolean })
                    }
                  />
                  <Label htmlFor="isActive">{t('فعال', 'Active')}</Label>
                </div>
              </div>

              {formData.image && (
                <div className="aspect-video rounded-lg overflow-hidden bg-muted">
                  <img src={formData.image} alt="" className="w-full h-full object-cover" />
                </div>
              )}

              <div className="flex justify-end gap-2">
                <Button type="button" variant="outline" onClick={() => setIsFormOpen(false)}>
                  {t('انصراف', 'Cancel')}
                </Button>
                <Button type="submit">
                  {editingBanner ? t('به‌روزرسانی', 'Update') : t('ذخیره', 'Save')}
                </Button>
              </div>
            </form>
          </DialogContent>
        </Dialog>
      </div>

      {/* Banners Table */}
      <div className="bg-card rounded-lg border overflow-hidden">
        <div className="overflow-x-auto">
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>{t('بنر', 'Banner')}</TableHead>
                <TableHead>{t('ترتیب', 'Order')}</TableHead>
                <TableHead>{t('وضعیت', 'Status')}</TableHead>
                <TableHead className="text-left">{t('عملیات', 'Actions')}</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {sortedBanners.length === 0 ? (
                <TableRow>
                  <TableCell colSpan={4} className="text-center py-8 text-muted-foreground">
                    {t('بنری یافت نشد', 'No banners found')}
                  </TableCell>
                </TableRow>
              ) : (
                sortedBanners.map((banner) => (
                  <TableRow key={banner.id}>
                    <TableCell>
                      <div className="flex items-center gap-3">
                        <div className="w-24 h-16 bg-muted rounded overflow-hidden">
                          <img
                            src={banner.image}
                            alt={language === 'fa' ? banner.title : banner.titleEn}
                            className="w-full h-full object-cover"
                          />
                        </div>
                        <div>
                          <p className="font-medium">
                            {language === 'fa' ? banner.title : banner.titleEn}
                          </p>
                          <p className="text-xs text-muted-foreground line-clamp-1">
                            {language === 'fa' ? banner.subtitle : banner.subtitleEn}
                          </p>
                        </div>
                      </div>
                    </TableCell>
                    <TableCell>{banner.sortOrder}</TableCell>
                    <TableCell>
                      {banner.isActive ? (
                        <span className="flex items-center gap-1 text-green-600">
                          <Eye className="h-4 w-4" />
                          {t('فعال', 'Active')}
                        </span>
                      ) : (
                        <span className="flex items-center gap-1 text-muted-foreground">
                          <EyeOff className="h-4 w-4" />
                          {t('غیرفعال', 'Inactive')}
                        </span>
                      )}
                    </TableCell>
                    <TableCell>
                      <div className="flex items-center gap-1">
                        <Button variant="ghost" size="icon" onClick={() => handleEdit(banner)}>
                          <Edit className="h-4 w-4" />
                        </Button>
                        <Button
                          variant="ghost"
                          size="icon"
                          className="text-destructive"
                          onClick={() => handleDelete(banner.id)}
                        >
                          <Trash2 className="h-4 w-4" />
                        </Button>
                      </div>
                    </TableCell>
                  </TableRow>
                ))
              )}
            </TableBody>
          </Table>
        </div>
      </div>
    </div>
  );
}
