import { useState } from 'react';
import { Link, useNavigate } from 'react-router-dom';
import { Eye, EyeOff, Lock, Mail, User, LogIn } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Separator } from '@/components/ui/separator';
import { useLanguage } from '@/contexts/LanguageContext';
import { useAuth } from '@/contexts/AuthContext';

export function Register() {
  const { t, dir } = useLanguage();
  const { register } = useAuth();
  const navigate = useNavigate();
  const [showPassword, setShowPassword] = useState(false);
  const [isLoading, setIsLoading] = useState(false);
  const [formData, setFormData] = useState({
    name: '',
    email: '',
    password: '',
    confirmPassword: '',
  });

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (formData.password !== formData.confirmPassword) {
      return;
    }

    setIsLoading(true);

    const success = await register(formData.name, formData.email, formData.password);
    if (success) {
      navigate('/');
    }

    setIsLoading(false);
  };

  return (
    <div className="min-h-screen flex items-center justify-center py-12 px-4">
      <div className="w-full max-w-md">
        <Card>
          <CardHeader className="text-center">
            <div className="w-16 h-16 bg-primary rounded-xl flex items-center justify-center mx-auto mb-4">
              <span className="text-primary-foreground font-bold text-2xl">D</span>
            </div>
            <CardTitle className="text-2xl">{t('ثبت‌نام', 'Register')}</CardTitle>
            <CardDescription>
              {t('حساب کاربری جدید بسازید', 'Create a new account')}
            </CardDescription>
          </CardHeader>

          <CardContent>
            <form onSubmit={handleSubmit} className="space-y-4" dir={dir}>
              <div className="space-y-2">
                <Label htmlFor="name">
                  <User className="h-4 w-4 inline ml-2" />
                  {t('نام و نام خانوادگی', 'Full Name')}
                </Label>
                <Input
                  id="name"
                  type="text"
                  value={formData.name}
                  onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                  placeholder={t('نام خود را وارد کنید', 'Enter your name')}
                  dir={dir}
                  required
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="email">
                  <Mail className="h-4 w-4 inline ml-2" />
                  {t('ایمیل', 'Email')}
                </Label>
                <Input
                  id="email"
                  type="email"
                  value={formData.email}
                  onChange={(e) => setFormData({ ...formData, email: e.target.value })}
                  placeholder={t('email@example.com', 'email@example.com')}
                  dir="ltr"
                  required
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="password">
                  <Lock className="h-4 w-4 inline ml-2" />
                  {t('رمز عبور', 'Password')}
                </Label>
                <div className="relative">
                  <Input
                    id="password"
                    type={showPassword ? 'text' : 'password'}
                    value={formData.password}
                    onChange={(e) => setFormData({ ...formData, password: e.target.value })}
                    placeholder={t('حداقل 6 کاراکتر', 'At least 6 characters')}
                    dir="ltr"
                    minLength={6}
                    required
                  />
                  <button
                    type="button"
                    onClick={() => setShowPassword(!showPassword)}
                    className="absolute right-3 top-1/2 -translate-y-1/2 text-muted-foreground hover:text-foreground"
                  >
                    {showPassword ? <EyeOff className="h-4 w-4" /> : <Eye className="h-4 w-4" />}
                  </button>
                </div>
              </div>

              <div className="space-y-2">
                <Label htmlFor="confirmPassword">
                  <Lock className="h-4 w-4 inline ml-2" />
                  {t('تکرار رمز عبور', 'Confirm Password')}
                </Label>
                <Input
                  id="confirmPassword"
                  type="password"
                  value={formData.confirmPassword}
                  onChange={(e) => setFormData({ ...formData, confirmPassword: e.target.value })}
                  placeholder={t('رمز عبور را تکرار کنید', 'Repeat password')}
                  dir="ltr"
                  required
                />
                {formData.password !== formData.confirmPassword && formData.confirmPassword && (
                  <p className="text-sm text-destructive">{t('رمز عبور مطابقت ندارد', 'Passwords do not match')}</p>
                )}
              </div>

              <div className="flex items-center gap-2 text-sm">
                <input type="checkbox" id="terms" className="rounded" required />
                <label htmlFor="terms">
                  {t('با', 'I agree to the')}{' '}
                  <Link to="/terms" className="text-primary hover:underline">
                    {t('قوانین و مقررات', 'Terms & Conditions')}
                  </Link>{' '}
                  {t('موافقم', 'agree')}
                </label>
              </div>

              <Button
                type="submit"
                className="w-full"
                size="lg"
                disabled={isLoading || formData.password !== formData.confirmPassword}
              >
                {isLoading ? (
                  <span className="flex items-center gap-2">
                    <span className="animate-spin">⏳</span>
                    {t('در حال ثبت‌نام...', 'Registering...')}
                  </span>
                ) : (
                  t('ثبت‌نام', 'Register')
                )}
              </Button>
            </form>

            <Separator className="my-6" />

            <div className="text-center">
              <p className="text-muted-foreground mb-4">
                {t('حساب کاربری دارید؟', 'Already have an account?')}
              </p>
              <Link to="/login">
                <Button variant="outline" className="w-full">
                  <LogIn className="h-4 w-4 ml-2" />
                  {t('ورود', 'Sign In')}
                </Button>
              </Link>
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}
