import { Link, useNavigate } from 'react-router-dom';
import { Trash2, Minus, Plus, ShoppingBag, ArrowLeft, ArrowRight } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Separator } from '@/components/ui/separator';
import { useLanguage } from '@/contexts/LanguageContext';
import { useCart } from '@/contexts/CartContext';
import { useAuth } from '@/contexts/AuthContext';
import { toast } from 'sonner';

export function Cart() {
  const { language, t, dir } = useLanguage();
  const { items, removeFromCart, updateQuantity, totalItems, totalPrice, clearCart } = useCart();
  const { isAuthenticated } = useAuth();
  const navigate = useNavigate();

  const formatPrice = (price: number) => {
    return new Intl.NumberFormat(language === 'fa' ? 'fa-IR' : 'en-US').format(price);
  };

  const handleCheckout = () => {
    if (!isAuthenticated) {
      toast.error(t('ابتدا وارد شوید', 'Please login first'));
      navigate('/login', { state: { from: '/checkout' } });
      return;
    }
    navigate('/checkout');
  };

  if (items.length === 0) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="text-center">
          <ShoppingBag className="h-24 w-24 mx-auto mb-6 text-muted-foreground" />
          <h1 className="text-2xl font-bold mb-4">{t('سبد خرید خالی است', 'Your cart is empty')}</h1>
          <p className="text-muted-foreground mb-6">{t('محصولی به سبد اضافه نکرده‌اید', 'You haven\'t added any products')}</p>
          <Link to="/products">
            <Button size="lg">
              {t('مشاهده محصولات', 'Browse Products')}
            </Button>
          </Link>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen py-8">
      <div className="container mx-auto px-4">
        <h1 className="text-2xl md:text-3xl font-bold mb-8">{t('سبد خرید', 'Shopping Cart')}</h1>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Cart Items */}
          <div className="lg:col-span-2 space-y-4">
            {items.map(({ product, quantity }) => (
              <Card key={product.id}>
                <CardContent className="p-4">
                  <div className="flex gap-4">
                    {/* Image */}
                    <Link to={`/product/${product.id}`} className="flex-shrink-0">
                      <div className="w-24 h-24 md:w-32 md:h-32 bg-muted rounded-lg overflow-hidden">
                        <img
                          src={product.images[0]}
                          alt={language === 'fa' ? product.name : product.nameEn}
                          className="w-full h-full object-cover"
                        />
                      </div>
                    </Link>

                    {/* Details */}
                    <div className="flex-1 min-w-0">
                      <Link to={`/product/${product.id}`}>
                        <h3 className="font-semibold text-lg mb-1 hover:text-primary transition-colors line-clamp-2">
                          {language === 'fa' ? product.name : product.nameEn}
                        </h3>
                      </Link>
                      <p className="text-sm text-muted-foreground mb-2">
                        {t('کد', 'SKU')}: {product.sku}
                      </p>

                      <div className="flex flex-wrap items-center justify-between gap-4">
                        {/* Quantity */}
                        <div className="flex items-center border rounded-md">
                          <button
                            className="px-3 py-1 hover:bg-muted transition-colors"
                            onClick={() => updateQuantity(product.id, quantity - 1)}
                            disabled={quantity <= 1}
                          >
                            <Minus className="h-4 w-4" />
                          </button>
                          <span className="px-3 py-1 font-medium min-w-[2rem] text-center">{quantity}</span>
                          <button
                            className="px-3 py-1 hover:bg-muted transition-colors"
                            onClick={() => updateQuantity(product.id, quantity + 1)}
                            disabled={quantity >= product.stock}
                          >
                            <Plus className="h-4 w-4" />
                          </button>
                        </div>

                        {/* Price */}
                        <div className="text-right">
                          <p className="font-bold text-lg text-primary">
                            {formatPrice(product.price * quantity)} {t('تومان', 'Toman')}
                          </p>
                          <p className="text-sm text-muted-foreground">
                            {formatPrice(product.price)} × {quantity}
                          </p>
                        </div>

                        {/* Remove */}
                        <Button
                          variant="ghost"
                          size="icon"
                          className="text-destructive hover:text-destructive"
                          onClick={() => removeFromCart(product.id)}
                        >
                          <Trash2 className="h-5 w-5" />
                        </Button>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>
            ))}

            <div className="flex justify-between items-center">
              <Button variant="outline" onClick={clearCart} className="text-destructive">
                <Trash2 className="h-4 w-4 ml-2" />
                {t('خالی کردن سبد', 'Clear Cart')}
              </Button>
              <Link to="/products">
                <Button variant="ghost">
                  {dir === 'rtl' ? <ArrowRight className="h-4 w-4 ml-2" /> : <ArrowLeft className="h-4 w-4 ml-2" />}
                  {t('ادامه خرید', 'Continue Shopping')}
                </Button>
              </Link>
            </div>
          </div>

          {/* Summary */}
          <div className="lg:col-span-1">
            <Card className="sticky top-24">
              <CardHeader>
                <CardTitle>{t('خلاصه سفارش', 'Order Summary')}</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex justify-between">
                  <span className="text-muted-foreground">
                    {t('تعداد محصولات', 'Items')} ({totalItems})
                  </span>
                  <span>{formatPrice(totalPrice)} {t('تومان', 'Toman')}</span>
                </div>

                <div className="flex justify-between">
                  <span className="text-muted-foreground">{t('هزینه ارسال', 'Shipping')}</span>
                  <span className="text-green-600">{t('رایگان', 'Free')}</span>
                </div>

                <Separator />

                <div className="flex justify-between text-lg font-bold">
                  <span>{t('جمع کل', 'Total')}</span>
                  <span className="text-primary">
                    {formatPrice(totalPrice)} {t('تومان', 'Toman')}
                  </span>
                </div>

                <Button className="w-full" size="lg" onClick={handleCheckout}>
                  {t('ادامه به پرداخت', 'Proceed to Checkout')}
                </Button>

                <p className="text-xs text-center text-muted-foreground">
                  {t('با ادامه، موافقت خود را با قوانین سایت اعلام می‌کنید', 'By continuing, you agree to our terms')}
                </p>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </div>
  );
}
