import { useState } from 'react';
import { Plus, Edit, Trash2, Eye, EyeOff } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Checkbox } from '@/components/ui/checkbox';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from '@/components/ui/dialog';
import { useLanguage } from '@/contexts/LanguageContext';
import { useData } from '@/contexts/DataContext';
import type { BlogPost } from '@/types';
import { toast } from 'sonner';

export function AdminBlog() {
  const { language, t, dir } = useLanguage();
  const { blogPosts, addBlogPost, updateBlogPost, deleteBlogPost } = useData();
  const [editingPost, setEditingPost] = useState<BlogPost | null>(null);
  const [isFormOpen, setIsFormOpen] = useState(false);

  const [formData, setFormData] = useState({
    title: '',
    titleEn: '',
    content: '',
    contentEn: '',
    excerpt: '',
    excerptEn: '',
    image: '',
    author: '',
    category: '',
    tags: [] as string[],
    isPublished: true,
  });
  const [newTag, setNewTag] = useState('');

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();

    if (!formData.title || !formData.content) {
      toast.error(t('لطفاً عنوان و محتوا را وارد کنید', 'Please enter title and content'));
      return;
    }

    if (editingPost) {
      updateBlogPost(editingPost.id, formData);
      toast.success(t('مقاله به‌روزرسانی شد', 'Post updated'));
    } else {
      addBlogPost(formData);
      toast.success(t('مقاله جدید اضافه شد', 'Post added'));
    }

    setIsFormOpen(false);
    setEditingPost(null);
    resetForm();
  };

  const resetForm = () => {
    setFormData({
      title: '',
      titleEn: '',
      content: '',
      contentEn: '',
      excerpt: '',
      excerptEn: '',
      image: '',
      author: '',
      category: '',
      tags: [],
      isPublished: true,
    });
  };

  const handleEdit = (post: BlogPost) => {
    setEditingPost(post);
    setFormData({
      title: post.title,
      titleEn: post.titleEn,
      content: post.content,
      contentEn: post.contentEn,
      excerpt: post.excerpt,
      excerptEn: post.excerptEn,
      image: post.image,
      author: post.author,
      category: post.category,
      tags: post.tags,
      isPublished: post.isPublished,
    });
    setIsFormOpen(true);
  };

  const handleDelete = (id: string) => {
    if (confirm(t('آیا از حذف این مقاله مطمئن هستید؟', 'Are you sure you want to delete this post?'))) {
      deleteBlogPost(id);
      toast.success(t('مقاله حذف شد', 'Post deleted'));
    }
  };

  const addTag = () => {
    if (newTag.trim() && !formData.tags.includes(newTag.trim())) {
      setFormData({ ...formData, tags: [...formData.tags, newTag.trim()] });
      setNewTag('');
    }
  };

  const removeTag = (tag: string) => {
    setFormData({ ...formData, tags: formData.tags.filter((t) => t !== tag) });
  };

  const sortedPosts = [...blogPosts].sort(
    (a, b) => new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime()
  );

  return (
    <div className="space-y-6">
      <div className="flex flex-col sm:flex-row sm:items-center justify-between gap-4">
        <h1 className="text-2xl md:text-3xl font-bold">{t('مدیریت وبلاگ', 'Manage Blog')}</h1>
        <Dialog open={isFormOpen} onOpenChange={setIsFormOpen}>
          <DialogTrigger asChild>
            <Button
              onClick={() => {
                setEditingPost(null);
                resetForm();
              }}
            >
              <Plus className="h-4 w-4 ml-2" />
              {t('افزودن مقاله', 'Add Post')}
            </Button>
          </DialogTrigger>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>
                {editingPost ? t('ویرایش مقاله', 'Edit Post') : t('افزودن مقاله جدید', 'Add New Post')}
              </DialogTitle>
              <DialogDescription>
                {t('اطلاعات مقاله را وارد کنید', 'Enter post information')}
              </DialogDescription>
            </DialogHeader>
            <form onSubmit={handleSubmit} className="space-y-4" dir={dir}>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="title">{t('عنوان (فارسی)', 'Title (FA)')} *</Label>
                  <Input
                    id="title"
                    value={formData.title}
                    onChange={(e) => setFormData({ ...formData, title: e.target.value })}
                    dir="rtl"
                    required
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="titleEn">{t('عنوان (انگلیسی)', 'Title (EN)')}</Label>
                  <Input
                    id="titleEn"
                    value={formData.titleEn}
                    onChange={(e) => setFormData({ ...formData, titleEn: e.target.value })}
                    dir="ltr"
                  />
                </div>
              </div>

              <div className="space-y-2">
                <Label htmlFor="excerpt">{t('خلاصه (فارسی)', 'Excerpt (FA)')}</Label>
                <Input
                  id="excerpt"
                  value={formData.excerpt}
                  onChange={(e) => setFormData({ ...formData, excerpt: e.target.value })}
                  dir="rtl"
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="excerptEn">{t('خلاصه (انگلیسی)', 'Excerpt (EN)')}</Label>
                <Input
                  id="excerptEn"
                  value={formData.excerptEn}
                  onChange={(e) => setFormData({ ...formData, excerptEn: e.target.value })}
                  dir="ltr"
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="content">{t('محتوا (فارسی)', 'Content (FA)')} *</Label>
                <textarea
                  id="content"
                  value={formData.content}
                  onChange={(e) => setFormData({ ...formData, content: e.target.value })}
                  className="w-full px-3 py-2 border rounded-md min-h-[200px]"
                  dir="rtl"
                  required
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="contentEn">{t('محتوا (انگلیسی)', 'Content (EN)')}</Label>
                <textarea
                  id="contentEn"
                  value={formData.contentEn}
                  onChange={(e) => setFormData({ ...formData, contentEn: e.target.value })}
                  className="w-full px-3 py-2 border rounded-md min-h-[200px]"
                  dir="ltr"
                />
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="image">{t('URL تصویر', 'Image URL')}</Label>
                  <Input
                    id="image"
                    value={formData.image}
                    onChange={(e) => setFormData({ ...formData, image: e.target.value })}
                    dir="ltr"
                    placeholder="https://..."
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="author">{t('نویسنده', 'Author')}</Label>
                  <Input
                    id="author"
                    value={formData.author}
                    onChange={(e) => setFormData({ ...formData, author: e.target.value })}
                    dir="rtl"
                  />
                </div>
              </div>

              <div className="space-y-2">
                <Label htmlFor="category">{t('دسته‌بندی', 'Category')}</Label>
                <Input
                  id="category"
                  value={formData.category}
                  onChange={(e) => setFormData({ ...formData, category: e.target.value })}
                  dir="rtl"
                />
              </div>

              <div className="space-y-2">
                <Label>{t('تگ‌ها', 'Tags')}</Label>
                <div className="flex gap-2">
                  <Input
                    value={newTag}
                    onChange={(e) => setNewTag(e.target.value)}
                    placeholder={t('تگ جدید', 'New tag')}
                    dir={dir}
                    onKeyPress={(e) => e.key === 'Enter' && (e.preventDefault(), addTag())}
                  />
                  <Button type="button" onClick={addTag}>
                    <Plus className="h-4 w-4" />
                  </Button>
                </div>
                <div className="flex flex-wrap gap-2 mt-2">
                  {formData.tags.map((tag) => (
                    <span
                      key={tag}
                      className="px-3 py-1 bg-muted rounded-full text-sm flex items-center gap-1"
                    >
                      {tag}
                      <button type="button" onClick={() => removeTag(tag)}>
                        <span className="text-muted-foreground">×</span>
                      </button>
                    </span>
                  ))}
                </div>
              </div>

              <div className="flex items-center gap-2">
                <Checkbox
                  id="isPublished"
                  checked={formData.isPublished}
                  onCheckedChange={(checked) =>
                    setFormData({ ...formData, isPublished: checked as boolean })
                  }
                />
                <Label htmlFor="isPublished">{t('منتشر شود', 'Published')}</Label>
              </div>

              <div className="flex justify-end gap-2">
                <Button type="button" variant="outline" onClick={() => setIsFormOpen(false)}>
                  {t('انصراف', 'Cancel')}
                </Button>
                <Button type="submit">
                  {editingPost ? t('به‌روزرسانی', 'Update') : t('ذخیره', 'Save')}
                </Button>
              </div>
            </form>
          </DialogContent>
        </Dialog>
      </div>

      {/* Posts Table */}
      <div className="bg-card rounded-lg border overflow-hidden">
        <div className="overflow-x-auto">
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>{t('مقاله', 'Post')}</TableHead>
                <TableHead>{t('نویسنده', 'Author')}</TableHead>
                <TableHead>{t('دسته‌بندی', 'Category')}</TableHead>
                <TableHead>{t('بازدید', 'Views')}</TableHead>
                <TableHead>{t('وضعیت', 'Status')}</TableHead>
                <TableHead className="text-left">{t('عملیات', 'Actions')}</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {sortedPosts.length === 0 ? (
                <TableRow>
                  <TableCell colSpan={6} className="text-center py-8 text-muted-foreground">
                    {t('مقاله‌ای یافت نشد', 'No posts found')}
                  </TableCell>
                </TableRow>
              ) : (
                sortedPosts.map((post) => (
                  <TableRow key={post.id}>
                    <TableCell>
                      <div className="flex items-center gap-3">
                        <div className="w-12 h-12 bg-muted rounded overflow-hidden">
                          <img
                            src={post.image}
                            alt={language === 'fa' ? post.title : post.titleEn}
                            className="w-full h-full object-cover"
                          />
                        </div>
                        <div>
                          <p className="font-medium line-clamp-1">
                            {language === 'fa' ? post.title : post.titleEn}
                          </p>
                          <p className="text-xs text-muted-foreground">
                            {new Date(post.createdAt).toLocaleDateString(
                              language === 'fa' ? 'fa-IR' : 'en-US'
                            )}
                          </p>
                        </div>
                      </div>
                    </TableCell>
                    <TableCell>{post.author}</TableCell>
                    <TableCell>{post.category}</TableCell>
                    <TableCell>{post.views}</TableCell>
                    <TableCell>
                      {post.isPublished ? (
                        <span className="flex items-center gap-1 text-green-600">
                          <Eye className="h-4 w-4" />
                          {t('منتشر شده', 'Published')}
                        </span>
                      ) : (
                        <span className="flex items-center gap-1 text-muted-foreground">
                          <EyeOff className="h-4 w-4" />
                          {t('پیش‌نویس', 'Draft')}
                        </span>
                      )}
                    </TableCell>
                    <TableCell>
                      <div className="flex items-center gap-1">
                        <Button variant="ghost" size="icon" onClick={() => handleEdit(post)}>
                          <Edit className="h-4 w-4" />
                        </Button>
                        <Button
                          variant="ghost"
                          size="icon"
                          className="text-destructive"
                          onClick={() => handleDelete(post.id)}
                        >
                          <Trash2 className="h-4 w-4" />
                        </Button>
                      </div>
                    </TableCell>
                  </TableRow>
                ))
              )}
            </TableBody>
          </Table>
        </div>
      </div>
    </div>
  );
}
