import { useState } from 'react';
import { useParams, Link } from 'react-router-dom';
import {
  Heart,
  Share2,
  ShoppingCart,
  Star,
  Check,
  X,
  Truck,
  Shield,
  RefreshCw,
  ChevronLeft,
  ChevronRight,
} from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Separator } from '@/components/ui/separator';
import { useLanguage } from '@/contexts/LanguageContext';
import { useCart } from '@/contexts/CartContext';
import { useData } from '@/contexts/DataContext';
import { ProductCard } from '@/components/ProductCard';
import { toast } from 'sonner';

export function ProductDetail() {
  const { id } = useParams<{ id: string }>();
  const { language, t, dir } = useLanguage();
  const { addToCart } = useCart();
  const { getProductById, products, categories } = useData();
  const [quantity, setQuantity] = useState(1);
  const [selectedImage, setSelectedImage] = useState(0);

  const product = getProductById(id || '');

  if (!product) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold mb-4">{t('محصول یافت نشد', 'Product not found')}</h1>
          <Link to="/products">
            <Button>{t('بازگشت به محصولات', 'Back to Products')}</Button>
          </Link>
        </div>
      </div>
    );
  }

  const category = categories.find((c) => c.id === product.category);
  const relatedProducts = products
    .filter((p) => p.category === product.category && p.id !== product.id)
    .slice(0, 4);

  const formatPrice = (price: number) => {
    return new Intl.NumberFormat(language === 'fa' ? 'fa-IR' : 'en-US').format(price);
  };

  const handleAddToCart = () => {
    if (product.stock > 0) {
      addToCart(product, quantity);
    }
  };

  const handleShare = () => {
    if (navigator.share) {
      navigator.share({
        title: language === 'fa' ? product.name : product.nameEn,
        url: window.location.href,
      });
    } else {
      navigator.clipboard.writeText(window.location.href);
      toast.success(t('لینک کپی شد', 'Link copied'));
    }
  };

  const nextImage = () => {
    setSelectedImage((prev) => (prev + 1) % product.images.length);
  };

  const prevImage = () => {
    setSelectedImage((prev) => (prev - 1 + product.images.length) % product.images.length);
  };

  return (
    <div className="min-h-screen py-8">
      <div className="container mx-auto px-4">
        {/* Breadcrumb */}
        <nav className="flex items-center gap-2 text-sm text-muted-foreground mb-6">
          <Link to="/" className="hover:text-primary">
            {t('صفحه اصلی', 'Home')}
          </Link>
          <ChevronLeft className="h-4 w-4" />
          <Link to="/products" className="hover:text-primary">
            {t('محصولات', 'Products')}
          </Link>
          {category && (
            <>
              <ChevronLeft className="h-4 w-4" />
              <Link to={`/category/${category.id}`} className="hover:text-primary">
                {language === 'fa' ? category.name : category.nameEn}
              </Link>
            </>
          )}
          <ChevronLeft className="h-4 w-4" />
          <span className="text-foreground">{language === 'fa' ? product.name : product.nameEn}</span>
        </nav>

        {/* Product Info */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8 mb-12">
          {/* Images */}
          <div className="space-y-4">
            <div className="relative aspect-square bg-muted rounded-xl overflow-hidden">
              <img
                src={product.images[selectedImage]}
                alt={language === 'fa' ? product.name : product.nameEn}
                className="w-full h-full object-cover"
              />
              {product.images.length > 1 && (
                <>
                  <button
                    onClick={prevImage}
                    className="absolute left-4 top-1/2 -translate-y-1/2 w-10 h-10 bg-white/90 rounded-full flex items-center justify-center hover:bg-white transition-colors"
                  >
                    <ChevronLeft className="h-5 w-5" />
                  </button>
                  <button
                    onClick={nextImage}
                    className="absolute right-4 top-1/2 -translate-y-1/2 w-10 h-10 bg-white/90 rounded-full flex items-center justify-center hover:bg-white transition-colors"
                  >
                    <ChevronRight className="h-5 w-5" />
                  </button>
                </>
              )}
              {/* Badges */}
              <div className="absolute top-4 left-4 flex flex-col gap-2">
                {product.isNew && <Badge className="bg-green-500">{t('جدید', 'New')}</Badge>}
                {product.discount && product.discount > 0 && (
                  <Badge variant="destructive">{product.discount}% {t('تخفیف', 'Off')}</Badge>
                )}
              </div>
            </div>

            {/* Thumbnails */}
            {product.images.length > 1 && (
              <div className="flex gap-2 overflow-x-auto">
                {product.images.map((image, index) => (
                  <button
                    key={index}
                    onClick={() => setSelectedImage(index)}
                    className={`flex-shrink-0 w-20 h-20 rounded-lg overflow-hidden border-2 ${
                      selectedImage === index ? 'border-primary' : 'border-transparent'
                    }`}
                  >
                    <img src={image} alt="" className="w-full h-full object-cover" />
                  </button>
                ))}
              </div>
            )}
          </div>

          {/* Details */}
          <div className="space-y-6">
            <div>
              <h1 className="text-2xl md:text-3xl font-bold mb-2">
                {language === 'fa' ? product.name : product.nameEn}
              </h1>
              <div className="flex items-center gap-4">
                <div className="flex items-center gap-1">
                  <Star className="h-5 w-5 fill-yellow-400 text-yellow-400" />
                  <span className="font-medium">{product.rating}</span>
                  <span className="text-muted-foreground">({product.reviewCount} {t('نظر', 'reviews')})</span>
                </div>
                <span className="text-muted-foreground">|</span>
                <span className="text-muted-foreground">
                  {t('کد محصول', 'SKU')}: {product.sku}
                </span>
              </div>
            </div>

            <Separator />

            {/* Price */}
            <div>
              <div className="flex items-center gap-4 mb-2">
                <span className="text-3xl font-bold text-primary">
                  {formatPrice(product.price)} {t('تومان', 'Toman')}
                </span>
                {product.originalPrice && product.originalPrice > product.price && (
                  <span className="text-xl text-muted-foreground line-through">
                    {formatPrice(product.originalPrice)}
                  </span>
                )}
              </div>
              {product.stock > 0 ? (
                <div className="flex items-center gap-2 text-green-600">
                  <Check className="h-5 w-5" />
                  <span>{t('موجود در انبار', 'In Stock')} ({product.stock} {t('عدد', 'units')})</span>
                </div>
              ) : (
                <div className="flex items-center gap-2 text-destructive">
                  <X className="h-5 w-5" />
                  <span>{t('ناموجود', 'Out of Stock')}</span>
                </div>
              )}
            </div>

            <Separator />

            {/* Description */}
            <p className="text-muted-foreground leading-relaxed">
              {language === 'fa' ? product.description : product.descriptionEn}
            </p>

            {/* Quantity and Add to Cart */}
            <div className="flex flex-wrap items-center gap-4">
              <div className="flex items-center border rounded-md">
                <button
                  className="px-4 py-2 hover:bg-muted transition-colors"
                  onClick={() => setQuantity(Math.max(1, quantity - 1))}
                  disabled={quantity <= 1}
                >
                  -
                </button>
                <span className="px-4 py-2 font-medium min-w-[3rem] text-center">{quantity}</span>
                <button
                  className="px-4 py-2 hover:bg-muted transition-colors"
                  onClick={() => setQuantity(Math.min(product.stock, quantity + 1))}
                  disabled={quantity >= product.stock}
                >
                  +
                </button>
              </div>

              <Button
                size="lg"
                onClick={handleAddToCart}
                disabled={product.stock === 0}
                className="flex-1 min-w-[200px]"
              >
                <ShoppingCart className="h-5 w-5 ml-2" />
                {product.stock === 0
                  ? t('ناموجود', 'Out of Stock')
                  : t('افزودن به سبد خرید', 'Add to Cart')}
              </Button>

              <Button variant="outline" size="icon" onClick={() => toast.success(t('به علاقه‌مندی‌ها اضافه شد', 'Added to favorites'))}>
                <Heart className="h-5 w-5" />
              </Button>

              <Button variant="outline" size="icon" onClick={handleShare}>
                <Share2 className="h-5 w-5" />
              </Button>
            </div>

            {/* Features */}
            <div className="grid grid-cols-3 gap-4 py-4">
              <div className="text-center">
                <Truck className="h-6 w-6 mx-auto mb-2 text-primary" />
                <p className="text-xs text-muted-foreground">{t('ارسال رایگان', 'Free Shipping')}</p>
              </div>
              <div className="text-center">
                <Shield className="h-6 w-6 mx-auto mb-2 text-primary" />
                <p className="text-xs text-muted-foreground">{t('گارانتی اصالت', 'Authenticity')}</p>
              </div>
              <div className="text-center">
                <RefreshCw className="h-6 w-6 mx-auto mb-2 text-primary" />
                <p className="text-xs text-muted-foreground">{t('۷ روز ضمانت بازگشت', '7 Days Return')}</p>
              </div>
            </div>
          </div>
        </div>

        {/* Tabs */}
        <Tabs defaultValue="specs" className="mb-12">
          <TabsList className="w-full justify-start">
            <TabsTrigger value="specs">{t('مشخصات فنی', 'Specifications')}</TabsTrigger>
            <TabsTrigger value="desc">{t('توضیحات', 'Description')}</TabsTrigger>
            <TabsTrigger value="reviews">{t('نظرات', 'Reviews')}</TabsTrigger>
          </TabsList>

          <TabsContent value="specs" className="mt-6">
            <div className="bg-muted rounded-lg p-6">
              <table className="w-full">
                <tbody>
                  {Object.entries(
                    language === 'fa' ? product.specifications : product.specificationsEn
                  ).map(([key, value]) => (
                    <tr key={key} className="border-b last:border-0">
                      <td className="py-3 font-medium w-1/3">{key}</td>
                      <td className="py-3 text-muted-foreground">{value}</td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </TabsContent>

          <TabsContent value="desc" className="mt-6">
            <div className="prose max-w-none">
              <p className="leading-relaxed">
                {language === 'fa' ? product.description : product.descriptionEn}
              </p>
            </div>
          </TabsContent>

          <TabsContent value="reviews" className="mt-6">
            <div className="text-center py-8">
              <p className="text-muted-foreground">{t('نظری ثبت نشده است', 'No reviews yet')}</p>
            </div>
          </TabsContent>
        </Tabs>

        {/* Related Products */}
        {relatedProducts.length > 0 && (
          <div>
            <h2 className="text-2xl font-bold mb-6">{t('محصولات مرتبط', 'Related Products')}</h2>
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6">
              {relatedProducts.map((product) => (
                <ProductCard key={product.id} product={product} />
              ))}
            </div>
          </div>
        )}
      </div>
    </div>
  );
}
