import { useParams, Link } from 'react-router-dom';
import { Calendar, User, Eye, ChevronLeft, Share2, Heart } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Separator } from '@/components/ui/separator';
import { useLanguage } from '@/contexts/LanguageContext';
import { useData } from '@/contexts/DataContext';
import { toast } from 'sonner';

export function BlogPostPage() {
  const { id } = useParams<{ id: string }>();
  const { language, t, dir } = useLanguage();
  const { getBlogPostById, blogPosts } = useData();

  const post = getBlogPostById(id || '');

  if (!post || !post.isPublished) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold mb-4">{t('مقاله یافت نشد', 'Article not found')}</h1>
          <Link to="/blog">
            <Button>{t('بازگشت به وبلاگ', 'Back to Blog')}</Button>
          </Link>
        </div>
      </div>
    );
  }

  const relatedPosts = blogPosts
    .filter((p) => p.id !== post.id && p.isPublished && p.category === post.category)
    .slice(0, 3);

  const formatDate = (date: string) => {
    return new Date(date).toLocaleDateString(language === 'fa' ? 'fa-IR' : 'en-US', {
      year: 'numeric',
      month: 'long',
      day: 'numeric',
    });
  };

  const handleShare = () => {
    if (navigator.share) {
      navigator.share({
        title: language === 'fa' ? post.title : post.titleEn,
        url: window.location.href,
      });
    } else {
      navigator.clipboard.writeText(window.location.href);
      toast.success(t('لینک کپی شد', 'Link copied'));
    }
  };

  // Convert markdown-like content to HTML
  const formatContent = (content: string) => {
    return content
      .split('\n\n')
      .map((paragraph) => {
        if (paragraph.startsWith('## ')) {
          return `<h2 class="text-2xl font-bold mt-8 mb-4">${paragraph.replace('## ', '')}</h2>`;
        }
        if (paragraph.startsWith('- ')) {
          const items = paragraph.split('\n').map((item) => `<li>${item.replace('- ', '')}</li>`).join('');
          return `<ul class="list-disc list-inside mb-4 space-y-1">${items}</ul>`;
        }
        return `<p class="mb-4 leading-relaxed">${paragraph}</p>`;
      })
      .join('');
  };

  return (
    <div className="min-h-screen py-8">
      <div className="container mx-auto px-4">
        {/* Breadcrumb */}
        <nav className="flex items-center gap-2 text-sm text-muted-foreground mb-6">
          <Link to="/" className="hover:text-primary">
            {t('صفحه اصلی', 'Home')}
          </Link>
          <ChevronLeft className="h-4 w-4" />
          <Link to="/blog" className="hover:text-primary">
            {t('وبلاگ', 'Blog')}
          </Link>
          <ChevronLeft className="h-4 w-4" />
          <span className="text-foreground line-clamp-1">
            {language === 'fa' ? post.title : post.titleEn}
          </span>
        </nav>

        {/* Header */}
        <div className="max-w-4xl mx-auto mb-8">
          <Badge className="mb-4">{post.category}</Badge>
          <h1 className="text-3xl md:text-4xl font-bold mb-6">
            {language === 'fa' ? post.title : post.titleEn}
          </h1>
          <div className="flex flex-wrap items-center gap-4 text-sm text-muted-foreground">
            <span className="flex items-center gap-1">
              <User className="h-4 w-4" />
              {post.author}
            </span>
            <span className="flex items-center gap-1">
              <Calendar className="h-4 w-4" />
              {formatDate(post.createdAt)}
            </span>
            <span className="flex items-center gap-1">
              <Eye className="h-4 w-4" />
              {post.views} {t('بازدید', 'views')}
            </span>
          </div>
        </div>

        {/* Featured Image */}
        <div className="max-w-4xl mx-auto mb-8">
          <div className="aspect-video rounded-xl overflow-hidden">
            <img
              src={post.image}
              alt={language === 'fa' ? post.title : post.titleEn}
              className="w-full h-full object-cover"
            />
          </div>
        </div>

        {/* Content */}
        <div className="max-w-4xl mx-auto">
          <div className="flex items-center justify-between mb-6">
            <div className="flex gap-2">
              {post.tags.map((tag) => (
                <Badge key={tag} variant="secondary">
                  #{tag}
                </Badge>
              ))}
            </div>
            <div className="flex gap-2">
              <Button
                variant="outline"
                size="icon"
                onClick={() => toast.success(t('به علاقه‌مندی‌ها اضافه شد', 'Added to favorites'))}
              >
                <Heart className="h-4 w-4" />
              </Button>
              <Button variant="outline" size="icon" onClick={handleShare}>
                <Share2 className="h-4 w-4" />
              </Button>
            </div>
          </div>

          <div
            className="prose prose-lg max-w-none dark:prose-invert"
            dangerouslySetInnerHTML={{
              __html: formatContent(language === 'fa' ? post.content : post.contentEn),
            }}
          />

          <Separator className="my-8" />

          {/* Author */}
          <div className="bg-muted rounded-xl p-6 mb-8">
            <div className="flex items-center gap-4">
              <div className="w-16 h-16 bg-primary rounded-full flex items-center justify-center">
                <User className="h-8 w-8 text-primary-foreground" />
              </div>
              <div>
                <h3 className="font-bold">{post.author}</h3>
                <p className="text-sm text-muted-foreground">
                  {t('نویسنده در فروشگاه دیجیتال', 'Author at Digital Shop')}
                </p>
              </div>
            </div>
          </div>
        </div>

        {/* Related Posts */}
        {relatedPosts.length > 0 && (
          <div className="max-w-4xl mx-auto mt-12">
            <h2 className="text-2xl font-bold mb-6">{t('مقالات مرتبط', 'Related Articles')}</h2>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
              {relatedPosts.map((relatedPost) => (
                <Link key={relatedPost.id} to={`/blog/${relatedPost.id}`}>
                  <article className="bg-card rounded-xl overflow-hidden border hover:shadow-lg transition-shadow">
                    <div className="aspect-video overflow-hidden">
                      <img
                        src={relatedPost.image}
                        alt={language === 'fa' ? relatedPost.title : relatedPost.titleEn}
                        className="w-full h-full object-cover hover:scale-105 transition-transform duration-300"
                      />
                    </div>
                    <div className="p-4">
                      <h3 className="font-bold line-clamp-2 hover:text-primary transition-colors">
                        {language === 'fa' ? relatedPost.title : relatedPost.titleEn}
                      </h3>
                    </div>
                  </article>
                </Link>
              ))}
            </div>
          </div>
        )}
      </div>
    </div>
  );
}
