import { useState } from 'react';
import { X, Plus } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Checkbox } from '@/components/ui/checkbox';
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from '@/components/ui/select';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { useLanguage } from '@/contexts/LanguageContext';
import { useData } from '@/contexts/DataContext';
import type { Product } from '@/types';
import { toast } from 'sonner';

interface ProductFormProps {
  product: Product | null;
  onClose: () => void;
}

export function ProductForm({ product, onClose }: ProductFormProps) {
  const { language, t, dir } = useLanguage();
  const { addProduct, updateProduct, categories } = useData();

  const [formData, setFormData] = useState<Partial<Product>>(
    product || {
      name: '',
      nameEn: '',
      description: '',
      descriptionEn: '',
      price: 0,
      originalPrice: undefined,
      images: [''],
      category: '',
      subcategory: '',
      stock: 0,
      sku: '',
      specifications: {},
      specificationsEn: {},
      rating: 5,
      reviewCount: 0,
      isFeatured: false,
      isNew: true,
      discount: undefined,
      tags: [],
    }
  );

  const [specs, setSpecs] = useState<[string, string][]>(
    Object.entries(product?.specifications || {})
  );
  const [specsEn, setSpecsEn] = useState<[string, string][]>(
    Object.entries(product?.specificationsEn || {})
  );
  const [newTag, setNewTag] = useState('');

  const activeCategories = categories.filter((c) => c.isActive);

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();

    if (!formData.name || !formData.price || !formData.category || !formData.sku) {
      toast.error(t('لطفاً همه فیلدهای ضروری را پر کنید', 'Please fill all required fields'));
      return;
    }

    const productData = {
      ...formData,
      specifications: Object.fromEntries(specs.filter(([k, v]) => k && v)),
      specificationsEn: Object.fromEntries(specsEn.filter(([k, v]) => k && v)),
      images: formData.images?.filter((img) => img) || [],
    };

    if (product) {
      updateProduct(product.id, productData);
      toast.success(t('محصول به‌روزرسانی شد', 'Product updated'));
    } else {
      addProduct(productData as Omit<Product, 'id' | 'createdAt' | 'updatedAt'>);
      toast.success(t('محصول جدید اضافه شد', 'Product added'));
    }

    onClose();
  };

  const addSpec = () => setSpecs([...specs, ['', '']]);
  const removeSpec = (index: number) => setSpecs(specs.filter((_, i) => i !== index));
  const updateSpec = (index: number, key: string, value: string) => {
    const newSpecs = [...specs];
    newSpecs[index] = [key, value];
    setSpecs(newSpecs);
  };

  const addSpecEn = () => setSpecsEn([...specsEn, ['', '']]);
  const removeSpecEn = (index: number) => setSpecsEn(specsEn.filter((_, i) => i !== index));
  const updateSpecEn = (index: number, key: string, value: string) => {
    const newSpecs = [...specsEn];
    newSpecs[index] = [key, value];
    setSpecsEn(newSpecs);
  };

  const addTag = () => {
    if (newTag.trim() && !formData.tags?.includes(newTag.trim())) {
      setFormData({ ...formData, tags: [...(formData.tags || []), newTag.trim()] });
      setNewTag('');
    }
  };

  const removeTag = (tag: string) => {
    setFormData({ ...formData, tags: formData.tags?.filter((t) => t !== tag) || [] });
  };

  const addImage = () => {
    setFormData({ ...formData, images: [...(formData.images || []), ''] });
  };

  const removeImage = (index: number) => {
    setFormData({
      ...formData,
      images: formData.images?.filter((_, i) => i !== index) || [],
    });
  };

  const updateImage = (index: number, value: string) => {
    const newImages = [...(formData.images || [])];
    newImages[index] = value;
    setFormData({ ...formData, images: newImages });
  };

  return (
    <form onSubmit={handleSubmit} className="space-y-6" dir={dir}>
      <Tabs defaultValue="basic">
        <TabsList className="w-full">
          <TabsTrigger value="basic">{t('اطلاعات اصلی', 'Basic Info')}</TabsTrigger>
          <TabsTrigger value="details">{t('جزئیات', 'Details')}</TabsTrigger>
          <TabsTrigger value="images">{t('تصاویر', 'Images')}</TabsTrigger>
          <TabsTrigger value="specs">{t('مشخصات', 'Specs')}</TabsTrigger>
        </TabsList>

        <TabsContent value="basic" className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="name">{t('نام محصول (فارسی)', 'Product Name (FA)')} *</Label>
              <Input
                id="name"
                value={formData.name}
                onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                dir="rtl"
                required
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="nameEn">{t('نام محصول (انگلیسی)', 'Product Name (EN)')}</Label>
              <Input
                id="nameEn"
                value={formData.nameEn}
                onChange={(e) => setFormData({ ...formData, nameEn: e.target.value })}
                dir="ltr"
              />
            </div>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="sku">{t('کد محصول (SKU)', 'SKU')} *</Label>
              <Input
                id="sku"
                value={formData.sku}
                onChange={(e) => setFormData({ ...formData, sku: e.target.value })}
                dir="ltr"
                required
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="category">{t('دسته‌بندی', 'Category')} *</Label>
              <Select
                value={formData.category}
                onValueChange={(value) => setFormData({ ...formData, category: value })}
              >
                <SelectTrigger>
                  <SelectValue placeholder={t('انتخاب دسته‌بندی', 'Select category')} />
                </SelectTrigger>
                <SelectContent>
                  {activeCategories.map((cat) => (
                    <SelectItem key={cat.id} value={cat.id}>
                      {language === 'fa' ? cat.name : cat.nameEn}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div className="space-y-2">
              <Label htmlFor="price">{t('قیمت', 'Price')} *</Label>
              <Input
                id="price"
                type="number"
                value={formData.price}
                onChange={(e) => setFormData({ ...formData, price: Number(e.target.value) })}
                dir="ltr"
                required
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="originalPrice">{t('قیمت اصلی', 'Original Price')}</Label>
              <Input
                id="originalPrice"
                type="number"
                value={formData.originalPrice || ''}
                onChange={(e) =>
                  setFormData({ ...formData, originalPrice: e.target.value ? Number(e.target.value) : undefined })
                }
                dir="ltr"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="stock">{t('موجودی', 'Stock')} *</Label>
              <Input
                id="stock"
                type="number"
                value={formData.stock}
                onChange={(e) => setFormData({ ...formData, stock: Number(e.target.value) })}
                dir="ltr"
                required
              />
            </div>
          </div>

          <div className="space-y-2">
            <Label htmlFor="description">{t('توضیحات (فارسی)', 'Description (FA)')}</Label>
            <textarea
              id="description"
              value={formData.description}
              onChange={(e) => setFormData({ ...formData, description: e.target.value })}
              className="w-full px-3 py-2 border rounded-md min-h-[100px]"
              dir="rtl"
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="descriptionEn">{t('توضیحات (انگلیسی)', 'Description (EN)')}</Label>
            <textarea
              id="descriptionEn"
              value={formData.descriptionEn}
              onChange={(e) => setFormData({ ...formData, descriptionEn: e.target.value })}
              className="w-full px-3 py-2 border rounded-md min-h-[100px]"
              dir="ltr"
            />
          </div>
        </TabsContent>

        <TabsContent value="details" className="space-y-4">
          <div className="flex gap-4">
            <div className="flex items-center gap-2">
              <Checkbox
                id="isNew"
                checked={formData.isNew}
                onCheckedChange={(checked) => setFormData({ ...formData, isNew: checked as boolean })}
              />
              <Label htmlFor="isNew">{t('محصول جدید', 'New Product')}</Label>
            </div>
            <div className="flex items-center gap-2">
              <Checkbox
                id="isFeatured"
                checked={formData.isFeatured}
                onCheckedChange={(checked) => setFormData({ ...formData, isFeatured: checked as boolean })}
              />
              <Label htmlFor="isFeatured">{t('محصول ویژه', 'Featured Product')}</Label>
            </div>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="rating">{t('امتیاز', 'Rating')}</Label>
              <Input
                id="rating"
                type="number"
                min="0"
                max="5"
                step="0.1"
                value={formData.rating}
                onChange={(e) => setFormData({ ...formData, rating: Number(e.target.value) })}
                dir="ltr"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="reviewCount">{t('تعداد نظرات', 'Review Count')}</Label>
              <Input
                id="reviewCount"
                type="number"
                value={formData.reviewCount}
                onChange={(e) => setFormData({ ...formData, reviewCount: Number(e.target.value) })}
                dir="ltr"
              />
            </div>
          </div>

          <div className="space-y-2">
            <Label>{t('تگ‌ها', 'Tags')}</Label>
            <div className="flex gap-2">
              <Input
                value={newTag}
                onChange={(e) => setNewTag(e.target.value)}
                placeholder={t('تگ جدید', 'New tag')}
                dir={dir}
                onKeyPress={(e) => e.key === 'Enter' && (e.preventDefault(), addTag())}
              />
              <Button type="button" onClick={addTag}>
                <Plus className="h-4 w-4" />
              </Button>
            </div>
            <div className="flex flex-wrap gap-2 mt-2">
              {formData.tags?.map((tag) => (
                <span
                  key={tag}
                  className="px-3 py-1 bg-muted rounded-full text-sm flex items-center gap-1"
                >
                  {tag}
                  <button type="button" onClick={() => removeTag(tag)}>
                    <X className="h-3 w-3" />
                  </button>
                </span>
              ))}
            </div>
          </div>
        </TabsContent>

        <TabsContent value="images" className="space-y-4">
          <div className="space-y-2">
            <Label>{t('تصاویر محصول', 'Product Images')}</Label>
            {formData.images?.map((image, index) => (
              <div key={index} className="flex gap-2">
                <Input
                  value={image}
                  onChange={(e) => updateImage(index, e.target.value)}
                  placeholder={t('URL تصویر', 'Image URL')}
                  dir="ltr"
                />
                <Button type="button" variant="ghost" size="icon" onClick={() => removeImage(index)}>
                  <X className="h-4 w-4" />
                </Button>
              </div>
            ))}
            <Button type="button" variant="outline" onClick={addImage} className="w-full">
              <Plus className="h-4 w-4 ml-2" />
              {t('افزودن تصویر', 'Add Image')}
            </Button>
          </div>

          {/* Image Preview */}
          {formData.images && formData.images.some((img) => img) && (
            <div className="grid grid-cols-4 gap-4 mt-4">
              {formData.images
                .filter((img) => img)
                .map((image, index) => (
                  <div key={index} className="aspect-square rounded-lg overflow-hidden bg-muted">
                    <img src={image} alt="" className="w-full h-full object-cover" />
                  </div>
                ))}
            </div>
          )}
        </TabsContent>

        <TabsContent value="specs" className="space-y-4">
          <div className="space-y-4">
            <div>
              <Label className="mb-2 block">{t('مشخصات فنی (فارسی)', 'Specifications (FA)')}</Label>
              {specs.map(([key, value], index) => (
                <div key={index} className="flex gap-2 mb-2">
                  <Input
                    value={key}
                    onChange={(e) => updateSpec(index, e.target.value, value)}
                    placeholder={t('نام', 'Name')}
                    dir="rtl"
                    className="flex-1"
                  />
                  <Input
                    value={value}
                    onChange={(e) => updateSpec(index, key, e.target.value)}
                    placeholder={t('مقدار', 'Value')}
                    dir="rtl"
                    className="flex-1"
                  />
                  <Button type="button" variant="ghost" size="icon" onClick={() => removeSpec(index)}>
                    <X className="h-4 w-4" />
                  </Button>
                </div>
              ))}
              <Button type="button" variant="outline" onClick={addSpec} className="w-full">
                <Plus className="h-4 w-4 ml-2" />
                {t('افزودن مشخصه', 'Add Spec')}
              </Button>
            </div>

            <div>
              <Label className="mb-2 block">{t('مشخصات فنی (انگلیسی)', 'Specifications (EN)')}</Label>
              {specsEn.map(([key, value], index) => (
                <div key={index} className="flex gap-2 mb-2">
                  <Input
                    value={key}
                    onChange={(e) => updateSpecEn(index, e.target.value, value)}
                    placeholder={t('نام', 'Name')}
                    dir="ltr"
                    className="flex-1"
                  />
                  <Input
                    value={value}
                    onChange={(e) => updateSpecEn(index, key, e.target.value)}
                    placeholder={t('مقدار', 'Value')}
                    dir="ltr"
                    className="flex-1"
                  />
                  <Button type="button" variant="ghost" size="icon" onClick={() => removeSpecEn(index)}>
                    <X className="h-4 w-4" />
                  </Button>
                </div>
              ))}
              <Button type="button" variant="outline" onClick={addSpecEn} className="w-full">
                <Plus className="h-4 w-4 ml-2" />
                {t('افزودن مشخصه', 'Add Spec')}
              </Button>
            </div>
          </div>
        </TabsContent>
      </Tabs>

      <div className="flex justify-end gap-2">
        <Button type="button" variant="outline" onClick={onClose}>
          {t('انصراف', 'Cancel')}
        </Button>
        <Button type="submit">
          {product ? t('به‌روزرسانی', 'Update') : t('ذخیره', 'Save')}
        </Button>
      </div>
    </form>
  );
}
